import tkinter as tk
from tkinter import ttk
import math

class CollisionSimulation:
    def __init__(self, root):
        self.root = root
        self.root.title("Исследование столкновений")
        self.root.geometry("1400x800")
        
        # Параметры системы
        self.collision_type = "Абсолютно упругое"
        self.mass1 = 1.0  # кг
        self.mass2 = 1.0  # кг
        self.velocity1 = 10.0  # м/с
        self.velocity2 = -5.0  # м/с
        self.distance = 200  # пикселей
        
        # Состояние симуляции
        self.simulating = False
        self.time_elapsed = 0
        self.collision_occurred = False
        
        # Позиции объектов
        self.object1_x = 100
        self.object2_x = 100 + self.distance
        self.object_width = 40
        self.ground_y = 500
        
        # Физические величины
        self.momentum1 = self.mass1 * self.velocity1
        self.momentum2 = self.mass2 * self.velocity2
        self.total_momentum = self.momentum1 + self.momentum2
        self.kinetic_energy1 = 0.5 * self.mass1 * self.velocity1**2
        self.kinetic_energy2 = 0.5 * self.mass2 * self.velocity2**2
        self.total_energy = self.kinetic_energy1 + self.kinetic_energy2
        
        self.setup_ui()
        self.calculate_initial_values()
        self.draw_scene()
    
    def setup_ui(self):
        # Основной фрейм
        main_frame = ttk.Frame(self.root, padding="10")
        main_frame.pack(fill=tk.BOTH, expand=True)
        
        # === Левая панель - управление ===
        left_frame = ttk.Frame(main_frame, width=300)
        left_frame.pack(side=tk.LEFT, fill=tk.Y, padx=(0, 10))
        left_frame.pack_propagate(False)
        
        ttk.Label(left_frame, text="Параметры системы", 
                 font=("Arial", 12, "bold")).pack(anchor=tk.W, pady=(0, 10))
        
        # Тип столкновения
        collision_frame = ttk.LabelFrame(left_frame, text="Тип столкновения", padding="5")
        collision_frame.pack(fill=tk.X, pady=5)
        
        self.collision_var = tk.StringVar(value=self.collision_type)
        collisions = ["Абсолютно упругое", "Абсолютно неупругое", "Упругое (коэффициент)"]
        
        for collision in collisions:
            ttk.Radiobutton(collision_frame, text=collision, variable=self.collision_var,
                           value=collision, command=self.on_collision_type_change).pack(anchor=tk.W)
        
        # Коэффициент восстановления (для упругих столкновений)
        self.restitution_frame = ttk.LabelFrame(left_frame, text="Коэффициент восстановления", padding="5")
        self.restitution_frame.pack(fill=tk.X, pady=5)
        
        ttk.Label(self.restitution_frame, text="e = 0.0-1.0:").pack(anchor=tk.W)
        self.restitution_var = tk.DoubleVar(value=0.8)
        restitution_scale = ttk.Scale(self.restitution_frame, from_=0.0, to=1.0,
                                     variable=self.restitution_var, orient=tk.HORIZONTAL)
        restitution_scale.pack(fill=tk.X, pady=2)
        
        self.restitution_value = ttk.Label(self.restitution_frame, text="0.8")
        self.restitution_value.pack(anchor=tk.W)
        
        restitution_scale.configure(command=self.on_restitution_change)
        
        # Масса объекта 1
        mass1_frame = ttk.LabelFrame(left_frame, text="Масса объекта 1", padding="5")
        mass1_frame.pack(fill=tk.X, pady=5)
        
        ttk.Label(mass1_frame, text="Масса (кг):").pack(anchor=tk.W)
        self.mass1_var = tk.DoubleVar(value=self.mass1)
        mass1_scale = ttk.Scale(mass1_frame, from_=0.1, to=10.0,
                               variable=self.mass1_var, orient=tk.HORIZONTAL)
        mass1_scale.pack(fill=tk.X, pady=2)
        
        self.mass1_value = ttk.Label(mass1_frame, text=f"{self.mass1} кг")
        self.mass1_value.pack(anchor=tk.W)
        
        mass1_scale.configure(command=self.on_mass1_change)
        
        # Масса объекта 2
        mass2_frame = ttk.LabelFrame(left_frame, text="Масса объекта 2", padding="5")
        mass2_frame.pack(fill=tk.X, pady=5)
        
        ttk.Label(mass2_frame, text="Масса (кг):").pack(anchor=tk.W)
        self.mass2_var = tk.DoubleVar(value=self.mass2)
        mass2_scale = ttk.Scale(mass2_frame, from_=0.1, to=10.0,
                               variable=self.mass2_var, orient=tk.HORIZONTAL)
        mass2_scale.pack(fill=tk.X, pady=2)
        
        self.mass2_value = ttk.Label(mass2_frame, text=f"{self.mass2} кг")
        self.mass2_value.pack(anchor=tk.W)
        
        mass2_scale.configure(command=self.on_mass2_change)
        
        # Скорость объекта 1
        velocity1_frame = ttk.LabelFrame(left_frame, text="Скорость объекта 1", padding="5")
        velocity1_frame.pack(fill=tk.X, pady=5)
        
        ttk.Label(velocity1_frame, text="Скорость (м/с):").pack(anchor=tk.W)
        self.velocity1_var = tk.DoubleVar(value=self.velocity1)
        velocity1_scale = ttk.Scale(velocity1_frame, from_=-20.0, to=20.0,
                                   variable=self.velocity1_var, orient=tk.HORIZONTAL)
        velocity1_scale.pack(fill=tk.X, pady=2)
        
        self.velocity1_value = ttk.Label(velocity1_frame, text=f"{self.velocity1} м/с")
        self.velocity1_value.pack(anchor=tk.W)
        
        velocity1_scale.configure(command=self.on_velocity1_change)
        
        # Скорость объекта 2
        velocity2_frame = ttk.LabelFrame(left_frame, text="Скорость объекта 2", padding="5")
        velocity2_frame.pack(fill=tk.X, pady=5)
        
        ttk.Label(velocity2_frame, text="Скорость (м/с):").pack(anchor=tk.W)
        self.velocity2_var = tk.DoubleVar(value=self.velocity2)
        velocity2_scale = ttk.Scale(velocity2_frame, from_=-20.0, to=20.0,
                                   variable=self.velocity2_var, orient=tk.HORIZONTAL)
        velocity2_scale.pack(fill=tk.X, pady=2)
        
        self.velocity2_value = ttk.Label(velocity2_frame, text=f"{self.velocity2} м/с")
        self.velocity2_value.pack(anchor=tk.W)
        
        velocity2_scale.configure(command=self.on_velocity2_change)
        
        # Расстояние между объектами
        distance_frame = ttk.LabelFrame(left_frame, text="Начальное расстояние", padding="5")
        distance_frame.pack(fill=tk.X, pady=5)
        
        ttk.Label(distance_frame, text="Расстояние (px):").pack(anchor=tk.W)
        self.distance_var = tk.IntVar(value=self.distance)
        distance_scale = ttk.Scale(distance_frame, from_=50, to=500,
                                  variable=self.distance_var, orient=tk.HORIZONTAL)
        distance_scale.pack(fill=tk.X, pady=2)
        
        self.distance_value = ttk.Label(distance_frame, text=f"{self.distance} px")
        self.distance_value.pack(anchor=tk.W)
        
        distance_scale.configure(command=self.on_distance_change)
        
        # Управление симуляцией
        control_frame = ttk.LabelFrame(left_frame, text="Управление", padding="5")
        control_frame.pack(fill=tk.X, pady=5)
        
        self.start_button = ttk.Button(control_frame, text="Запуск", 
                                      command=self.start_simulation)
        self.start_button.pack(fill=tk.X, pady=2)
        
        self.stop_button = ttk.Button(control_frame, text="Стоп", 
                                     command=self.stop_simulation, state=tk.DISABLED)
        self.stop_button.pack(fill=tk.X, pady=2)
        
        self.reset_button = ttk.Button(control_frame, text="Сброс", 
                                      command=self.reset_simulation)
        self.reset_button.pack(fill=tk.X, pady=2)
        
        # Информация о типах столкновений
        info_frame = ttk.LabelFrame(left_frame, text="Типы столкновений", padding="5")
        info_frame.pack(fill=tk.X, pady=5)
        
        info_text = """Абсолютно упругое:
• Сохраняется энергия
• Сохраняется импульс
• Объекты разлетаются

Абсолютно неупругое:
• Макс. потеря энергии
• Объекты слипаются
• Сохраняется импульс

Упругое (коэффициент):
• Частичная потеря энергии
• Коэффициент e = 0-1"""
        
        info_label = ttk.Label(info_frame, text=info_text, font=("Arial", 8),
                              justify=tk.LEFT)
        info_label.pack(anchor=tk.W)
        
        # === Центральная панель - визуализация ===
        center_frame = ttk.Frame(main_frame)
        center_frame.pack(side=tk.LEFT, fill=tk.BOTH, expand=True, padx=10)
        
        # Холст для отрисовки
        self.canvas = tk.Canvas(center_frame, bg="white", highlightthickness=1, 
                               highlightbackground="gray")
        self.canvas.pack(fill=tk.BOTH, expand=True)
        
        # === Правая панель - текущие значения ===
        right_frame = ttk.Frame(main_frame, width=350)
        right_frame.pack(side=tk.RIGHT, fill=tk.Y, padx=(10, 0))
        right_frame.pack_propagate(False)
        
        ttk.Label(right_frame, text="Текущие значения", 
                 font=("Arial", 12, "bold")).pack(anchor=tk.W, pady=(0, 10))
        
        # Объект 1
        obj1_frame = ttk.LabelFrame(right_frame, text="Объект 1", padding="5")
        obj1_frame.pack(fill=tk.X, pady=5)
        
        self.obj1_mass_label = ttk.Label(obj1_frame, text="Масса: 1.0 кг", font=("Arial", 10))
        self.obj1_mass_label.pack(anchor=tk.W)
        
        self.obj1_velocity_label = ttk.Label(obj1_frame, text="Скорость: 0.00 м/с", font=("Arial", 10))
        self.obj1_velocity_label.pack(anchor=tk.W)
        
        self.obj1_momentum_label = ttk.Label(obj1_frame, text="Импульс: 0.00 кг·м/с", font=("Arial", 10))
        self.obj1_momentum_label.pack(anchor=tk.W)
        
        self.obj1_energy_label = ttk.Label(obj1_frame, text="Энергия: 0.00 Дж", font=("Arial", 10))
        self.obj1_energy_label.pack(anchor=tk.W)
        
        # Объект 2
        obj2_frame = ttk.LabelFrame(right_frame, text="Объект 2", padding="5")
        obj2_frame.pack(fill=tk.X, pady=5)
        
        self.obj2_mass_label = ttk.Label(obj2_frame, text="Масса: 1.0 кг", font=("Arial", 10))
        self.obj2_mass_label.pack(anchor=tk.W)
        
        self.obj2_velocity_label = ttk.Label(obj2_frame, text="Скорость: 0.00 м/с", font=("Arial", 10))
        self.obj2_velocity_label.pack(anchor=tk.W)
        
        self.obj2_momentum_label = ttk.Label(obj2_frame, text="Импульс: 0.00 кг·м/с", font=("Arial", 10))
        self.obj2_momentum_label.pack(anchor=tk.W)
        
        self.obj2_energy_label = ttk.Label(obj2_frame, text="Энергия: 0.00 Дж", font=("Arial", 10))
        self.obj2_energy_label.pack(anchor=tk.W)
        
        # Система в целом
        system_frame = ttk.LabelFrame(right_frame, text="Система", padding="5")
        system_frame.pack(fill=tk.X, pady=5)
        
        self.total_momentum_label = ttk.Label(system_frame, text="Суммарный импульс: 0.00 кг·м/с", 
                                            font=("Arial", 10, "bold"))
        self.total_momentum_label.pack(anchor=tk.W)
        
        self.total_energy_label = ttk.Label(system_frame, text="Суммарная энергия: 0.00 Дж", 
                                          font=("Arial", 10, "bold"))
        self.total_energy_label.pack(anchor=tk.W)
        
        self.energy_loss_label = ttk.Label(system_frame, text="Потери энергии: 0.00%", 
                                         font=("Arial", 10))
        self.energy_loss_label.pack(anchor=tk.W)
        
        # Статус столкновения
        status_frame = ttk.LabelFrame(right_frame, text="Статус", padding="5")
        status_frame.pack(fill=tk.X, pady=5)
        
        self.collision_status_label = ttk.Label(status_frame, text="Столкновение: НЕТ", 
                                              font=("Arial", 10))
        self.collision_status_label.pack(anchor=tk.W)
        
        self.time_label = ttk.Label(status_frame, text="Время: 0.00 с", font=("Arial", 10))
        self.time_label.pack(anchor=tk.W)
        
        # Теоретические расчеты
        theory_frame = ttk.LabelFrame(right_frame, text="Теоретические значения", padding="5")
        theory_frame.pack(fill=tk.X, pady=5)
        
        self.final_velocity1_label = ttk.Label(theory_frame, text="V1 после: -", font=("Arial", 9))
        self.final_velocity1_label.pack(anchor=tk.W)
        
        self.final_velocity2_label = ttk.Label(theory_frame, text="V2 после: -", font=("Arial", 9))
        self.final_velocity2_label.pack(anchor=tk.W)
        
        self.update_ui_elements()
    
    def on_collision_type_change(self):
        self.collision_type = self.collision_var.get()
        if self.collision_type == "Упругое (коэффициент)":
            self.restitution_frame.pack(fill=tk.X, pady=5)
        else:
            self.restitution_frame.pack_forget()
        
        self.calculate_theoretical_values()
        self.draw_scene()
    
    def on_restitution_change(self, event=None):
        self.restitution_value.config(text=f"{self.restitution_var.get():.2f}")
        self.calculate_theoretical_values()
    
    def on_mass1_change(self, event=None):
        self.mass1 = self.mass1_var.get()
        self.mass1_value.config(text=f"{self.mass1:.1f} кг")
        self.calculate_initial_values()
        self.calculate_theoretical_values()
        self.draw_scene()
    
    def on_mass2_change(self, event=None):
        self.mass2 = self.mass2_var.get()
        self.mass2_value.config(text=f"{self.mass2:.1f} кг")
        self.calculate_initial_values()
        self.calculate_theoretical_values()
        self.draw_scene()
    
    def on_velocity1_change(self, event=None):
        self.velocity1 = self.velocity1_var.get()
        self.velocity1_value.config(text=f"{self.velocity1:.1f} м/с")
        self.calculate_initial_values()
        self.calculate_theoretical_values()
        self.draw_scene()
    
    def on_velocity2_change(self, event=None):
        self.velocity2 = self.velocity2_var.get()
        self.velocity2_value.config(text=f"{self.velocity2:.1f} м/с")
        self.calculate_initial_values()
        self.calculate_theoretical_values()
        self.draw_scene()
    
    def on_distance_change(self, event=None):
        self.distance = self.distance_var.get()
        self.distance_value.config(text=f"{self.distance} px")
        self.reset_positions()
        self.draw_scene()
    
    def calculate_initial_values(self):
        """Расчет начальных физических величин"""
        self.momentum1 = self.mass1 * self.velocity1
        self.momentum2 = self.mass2 * self.velocity2
        self.total_momentum = self.momentum1 + self.momentum2
        self.kinetic_energy1 = 0.5 * self.mass1 * self.velocity1**2
        self.kinetic_energy2 = 0.5 * self.mass2 * self.velocity2**2
        self.total_energy = self.kinetic_energy1 + self.kinetic_energy2
        
        self.update_ui_elements()
    
    def calculate_theoretical_values(self):
        """Расчет теоретических значений после столкновения"""
        m1, m2 = self.mass1, self.mass2
        v1, v2 = self.velocity1, self.velocity2
        
        if self.collision_type == "Абсолютно упругое":
            # Формулы для абсолютно упругого столкновения
            v1_final = ((m1 - m2) * v1 + 2 * m2 * v2) / (m1 + m2)
            v2_final = ((m2 - m1) * v2 + 2 * m1 * v1) / (m1 + m2)
            
        elif self.collision_type == "Абсолютно неупругое":
            # Общая скорость после столкновения
            v_common = (m1 * v1 + m2 * v2) / (m1 + m2)
            v1_final = v_common
            v2_final = v_common
            
        else:  # Упругое с коэффициентом
            e = self.restitution_var.get()
            v1_final = ((m1 - e * m2) * v1 + (1 + e) * m2 * v2) / (m1 + m2)
            v2_final = ((m2 - e * m1) * v2 + (1 + e) * m1 * v1) / (m1 + m2)
        
        self.final_velocity1_label.config(text=f"V1 после: {v1_final:.2f} м/с")
        self.final_velocity2_label.config(text=f"V2 после: {v2_final:.2f} м/с")
    
    def update_ui_elements(self):
        """Обновление всех UI элементов"""
        # Объект 1
        self.obj1_mass_label.config(text=f"Масса: {self.mass1:.1f} кг")
        self.obj1_velocity_label.config(text=f"Скорость: {self.velocity1:.2f} м/с")
        self.obj1_momentum_label.config(text=f"Импульс: {self.momentum1:.2f} кг·м/с")
        self.obj1_energy_label.config(text=f"Энергия: {self.kinetic_energy1:.2f} Дж")
        
        # Объект 2
        self.obj2_mass_label.config(text=f"Масса: {self.mass2:.1f} кг")
        self.obj2_velocity_label.config(text=f"Скорость: {self.velocity2:.2f} м/с")
        self.obj2_momentum_label.config(text=f"Импульс: {self.momentum2:.2f} кг·м/с")
        self.obj2_energy_label.config(text=f"Энергия: {self.kinetic_energy2:.2f} Дж")
        
        # Система
        self.total_momentum_label.config(text=f"Суммарный импульс: {self.total_momentum:.2f} кг·м/с")
        self.total_energy_label.config(text=f"Суммарная энергия: {self.total_energy:.2f} Дж")
        self.energy_loss_label.config(text="Потери энергии: 0.00%")
    
    def reset_positions(self):
        """Сброс позиций объектов"""
        self.object1_x = 100
        self.object2_x = 100 + self.distance
        self.collision_occurred = False
    
    def start_simulation(self):
        if not self.simulating:
            self.simulating = True
            self.time_elapsed = 0
            self.collision_occurred = False
            self.start_button.config(state=tk.DISABLED)
            self.stop_button.config(state=tk.NORMAL)
            
            self.root.after(10, self.run_simulation_step)
    
    def stop_simulation(self):
        self.simulating = False
        self.start_button.config(state=tk.NORMAL)
        self.stop_button.config(state=tk.DISABLED)
    
    def reset_simulation(self):
        self.simulating = False
        self.time_elapsed = 0
        self.collision_occurred = False
        self.start_button.config(state=tk.NORMAL)
        self.stop_button.config(state=tk.DISABLED)
        self.reset_positions()
        self.calculate_initial_values()
        self.draw_scene()
        self.collision_status_label.config(text="Столкновение: НЕТ")
        self.time_label.config(text="Время: 0.00 с")
    
    def run_simulation_step(self):
        """Выполнение одного шага симуляции"""
        if not self.simulating:
            return
        
        time_step = 0.05  # 50 мс
        scale = 10  # пикселей на метр
        
        # Обновление позиций
        self.object1_x += self.velocity1 * scale * time_step
        self.object2_x += self.velocity2 * scale * time_step
        
        # Проверка столкновения
        if not self.collision_occurred:
            distance_between = abs(self.object2_x - self.object1_x)
            if distance_between <= self.object_width:
                self.handle_collision()
                self.collision_occurred = True
                self.collision_status_label.config(text="Столкновение: ПРОИЗОШЛО!")
        
        # Обновление времени
        self.time_elapsed += time_step
        self.time_label.config(text=f"Время: {self.time_elapsed:.2f} с")
        
        # Обновление физических величин
        self.update_physical_values()
        
        # Отрисовка
        self.draw_scene()
        
        # Продолжение симуляции
        self.root.after(50, self.run_simulation_step)  # 20 FPS
    
    def handle_collision(self):
        """Обработка столкновения"""
        m1, m2 = self.mass1, self.mass2
        v1, v2 = self.velocity1, self.velocity2
        
        if self.collision_type == "Абсолютно упругое":
            # Абсолютно упругое столкновение
            self.velocity1 = ((m1 - m2) * v1 + 2 * m2 * v2) / (m1 + m2)
            self.velocity2 = ((m2 - m1) * v2 + 2 * m1 * v1) / (m1 + m2)
            
        elif self.collision_type == "Абсолютно неупругое":
            # Абсолютно неупругое столкновение
            v_common = (m1 * v1 + m2 * v2) / (m1 + m2)
            self.velocity1 = v_common
            self.velocity2 = v_common
            
        else:  # Упругое с коэффициентом
            e = self.restitution_var.get()
            self.velocity1 = ((m1 - e * m2) * v1 + (1 + e) * m2 * v2) / (m1 + m2)
            self.velocity2 = ((m2 - e * m1) * v2 + (1 + e) * m1 * v1) / (m1 + m2)
        
        # Корректировка позиций чтобы избежать наложения
        self.object1_x = min(self.object1_x, self.object2_x - self.object_width)
        self.object2_x = max(self.object2_x, self.object1_x + self.object_width)
    
    def update_physical_values(self):
        """Обновление физических величин во время симуляции"""
        self.momentum1 = self.mass1 * self.velocity1
        self.momentum2 = self.mass2 * self.velocity2
        self.total_momentum = self.momentum1 + self.momentum2
        self.kinetic_energy1 = 0.5 * self.mass1 * self.velocity1**2
        self.kinetic_energy2 = 0.5 * self.mass2 * self.velocity2**2
        
        new_total_energy = self.kinetic_energy1 + self.kinetic_energy2
        energy_loss = ((self.total_energy - new_total_energy) / self.total_energy * 100) if self.total_energy > 0 else 0
        
        self.total_energy = new_total_energy
        
        # Обновление UI
        self.obj1_velocity_label.config(text=f"Скорость: {self.velocity1:.2f} м/с")
        self.obj2_velocity_label.config(text=f"Скорость: {self.velocity2:.2f} м/с")
        
        self.obj1_momentum_label.config(text=f"Импульс: {self.momentum1:.2f} кг·м/с")
        self.obj2_momentum_label.config(text=f"Импульс: {self.momentum2:.2f} кг·м/с")
        
        self.obj1_energy_label.config(text=f"Энергия: {self.kinetic_energy1:.2f} Дж")
        self.obj2_energy_label.config(text=f"Энергия: {self.kinetic_energy2:.2f} Дж")
        
        self.total_momentum_label.config(text=f"Суммарный импульс: {self.total_momentum:.2f} кг·м/с")
        self.total_energy_label.config(text=f"Суммарная энергия: {self.total_energy:.2f} Дж")
        self.energy_loss_label.config(text=f"Потери энергии: {energy_loss:.2f}%")
    
    def draw_scene(self):
        """Отрисовка сцены"""
        self.canvas.delete("all")
        
        canvas_width = self.canvas.winfo_width()
        canvas_height = self.canvas.winfo_height()
        
        if canvas_width <= 1 or canvas_height <= 1:
            return
        
        # Земля
        ground_y = self.ground_y
        self.canvas.create_line(0, ground_y, canvas_width, ground_y, fill="black", width=2)
        
        # Объект 1 (синий)
        obj1_color = "blue"
        obj1_y = ground_y - self.object_width
        self.canvas.create_rectangle(
            self.object1_x - self.object_width//2, obj1_y - self.object_width,
            self.object1_x + self.object_width//2, obj1_y,
            fill=obj1_color, outline="darkblue", width=2
        )
        
        # Объект 2 (красный)
        obj2_color = "red"
        obj2_y = ground_y - self.object_width
        self.canvas.create_rectangle(
            self.object2_x - self.object_width//2, obj2_y - self.object_width,
            self.object2_x + self.object_width//2, obj2_y,
            fill=obj2_color, outline="darkred", width=2
        )
        
        # Векторы скорости
        velocity_scale = 5
        self.draw_velocity_vector(self.object1_x, obj1_y, self.velocity1, obj1_color, velocity_scale)
        self.draw_velocity_vector(self.object2_x, obj2_y, self.velocity2, obj2_color, velocity_scale)
        
        # Подписи
        self.canvas.create_text(self.object1_x, obj1_y - self.object_width - 10, 
                              text=f"m1={self.mass1}кг", font=("Arial", 10), fill=obj1_color)
        self.canvas.create_text(self.object2_x, obj2_y - self.object_width - 10, 
                              text=f"m2={self.mass2}кг", font=("Arial", 10), fill=obj2_color)
        
        # Информация о столкновении
        if self.collision_occurred:
            self.canvas.create_text(canvas_width//2, 30, 
                                  text="СТОЛКНОВЕНИЕ ПРОИЗОШЛО!", 
                                  font=("Arial", 14, "bold"), fill="red")
        
        # Тип столкновения
        self.canvas.create_text(canvas_width//2, 60, 
                              text=f"Тип: {self.collision_type}", 
                              font=("Arial", 12), fill="darkgreen")
    
    def draw_velocity_vector(self, x, y, velocity, color, scale):
        """Отрисовка вектора скорости"""
        if abs(velocity) > 0.1:  # Рисуем только если скорость значительная
            arrow_length = velocity * scale
            arrow_end_x = x + arrow_length
            
            self.canvas.create_line(x, y, arrow_end_x, y, 
                                  arrow=tk.LAST, fill=color, width=3)
            
            # Подпись скорости
            self.canvas.create_text(arrow_end_x + 20, y, 
                                  text=f"{velocity:.1f} м/с", 
                                  font=("Arial", 8), fill=color)

def main():
    root = tk.Tk()
    app = CollisionSimulation(root)
    root.mainloop()

if __name__ == "__main__":
    main()
